function [out,fmin,parout] =sfista(Ffun,Ffun_grad,Gfun,Gfun_prox,Hfun,Hfun_prox,Afun,Atfun,mu,lambda_G,lambda_H,startx,par)
%SFISTA employs the smooth fista method for solving the problem min{ f(x) + lambda_G * g(A(x)) + lambda_H * h(x) }
%
% Underlying assumptions:
% All functions are convex
% f is smooth
% g ia real-valued and proximable
% h is proper closed and proximable
% A is a linear transformation
% lambda_G and lambda_H are positive scalars
% ====================================================
% Usage:
% out               = SFISTA(Ffun,Ffun_grad,Gfun,Gfun_prox,Hfun,Hfun_prox,Afun,Atfun,mu,lambda_G,lambda_H,startx,par)
% [out,fmin]        = SFISTA(Ffun,Ffun_grad,Gfun,Gfun_prox,Hfun,Hfun_prox,Afun,Atfun,mu,lambda_G,lambda_H,startx,par)
% [out,fmin,parout] = SFISTA(Ffun,Ffun_grad,Gfun,Gfun_prox,Hfun,Hfun_prox,Afun,Atfun,mu,lambda_G,lambda_H,startx,par)
% ====================================================
% Input:
% Ffun        - function handle for the function f
% Ffun_grad   - function handle for the gradient of the function f
% Gfun        - function handle for the function g
% Gfun_prox   - function handle for the proximal mapping of g times a postive constant
% Hfun        - function handle for the function h
% Hfun_prox   - function handle for the proximal mapping of h times a postive constant
% Afun        - function handle for the linear transformation of A
% Atfun       - function handle for the adjoint of the linear transformation of A
% mu          - positive smoothing parameter
% lambda_G    - positive scalar penalty for the function g
% lambda_H    - positive scalar penalty for the function h
% startx      - starting vector
% par         - struct which contains different values required for the operation of SFISTA
% Fields of par:
%       max_iter      - maximal number of iterations [default: 1000]
%       eco_flag      - true if economic version (without calculating objective function
%                              values) should run, otherwise false [default: false]
%       print_flag    - true if internal printing should take place, otherwise false [default: true]
%       monotone_flag - true if monotonic method should be applied, otherwise false [default: false]
%       Lstart        - initial value for the Lipschitz constant (scalar) [default: 1]
%       const_flag    - true if constant step size should be used, otherwise false [default: false]
%       regret_flag   - true if the Lipschitz constant should be divided by eta in the next iteration [default: false]
%       eta           - the multiplicative constant used when regretting or backtracking [default: 2]
%       eps           - stopping criteria tolerance (the method stops when the
%                       norm of the difference between consecutive iterates is < eps) [default: 1e-5]
%                       relevant only in non monotonic method
%       continuation  - true if continuation mode should be used, otherwise false [default: false]
%       gamma         - multiplicative constant used to increase mu. [default: 5]
%                       Relevant only in continuation mode.
%       cont_cycles   - number of cycles in continuation mode [default: 5]
%                       Relevant only in continuation mode.r
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    LValVec   - vector of all used Lipschitz constants (when par.const_flag is false) 
%    OrigFun   - original function value (without smoothing)
%    CycleIter - number of performed iterations in each cycle (relevant only in continuation mode)

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

if ((nargin < 12)  || ( isa (Ffun_grad,'function_handle')~= 1) || ( isa (Gfun_prox,'function_handle')~= 1) || ...
        ( isa (Hfun_prox,'function_handle')~= 1) || ...
        ( isa (Afun,'function_handle')~= 1) || ( isa (Atfun,'function_handle')~= 1))
    error ('usage: sfista(F_handle,F_Grad_handle,G_handle,Prox_G_handle,H_handle,Prox_H_handle,Afun_handle,Atfun_handle,mu,lamdaG,lamdaH,startpoint,[par]) ') ;
end



    function gmu = gu_fun(x)
        gmu = Gfun( Gfun_prox(x,mu)) + 1/(2 * mu) * norm(Gfun_prox(x,mu) - x,'fro') ^2 ;
    end

    function res = combinedf( x)
        res =  Ffun(x) +  lambda_G * gu_fun(Afun(x)) ;
    end

    function grad_mu = grad_gu_fun(x)
        grad_mu = 1 / mu * (x - Gfun_prox(x,mu)) ;
    end

    function grad = combined_grad(x)
        grad = Ffun_grad(x) + lambda_G * Atfun(grad_gu_fun (Afun(x))) ;
    end


if (nargin < 13)
    par = struct() ;
end

if ( isfield (par,'print_flag') && par.print_flag)
    fprintf('*********************\n');
    fprintf('SFISTA\n')
    fprintf('Preparing parameters for fista\n')
    fprintf('*********************\n');
end

par = initialize_par_for_prox_gradient(par,'sfista') ;
mult_eps_const = 100 ; % for inner cycles

if (par.continuation && (nargout > 2))
    CycleIter = zeros(par.cont_cycles,1) ;
end

if ( isempty(Ffun) || isempty(Gfun) || isempty(Hfun) ||  ~isa(Ffun,'function_handle') || ~isa(Gfun,'function_handle') || ~isa(Hfun,'function_handle') )
    if (nargout > 1)
        error('When FFun, Gfun or Hfun is not a function handle usage: out = sfista([],Ffun_grad,[],Gfun_prox,[],Hfun_prox,Afun,Atfun,mu,lambda_G,lambda_H,startx,[par])')
    else
        fprintf('Ffun, Gfun or Hfun is not a function handle, changing to economical mode\n') ;
        par.eco_flag = true ;
        par.print_flag = false ;
        par.const_flag = true ;
        par.monotone_flag = false ;
        
        if (par.continuation)
            save_mu = mu ;
            save_eps = par.eps ;
            next_start = startx ;
            par.eps = par.eps * mult_eps_const ; %inner cycles with bigger eps

            for k = par.cont_cycles-1:-1:1
                mu =  save_mu * par.gamma ^ k ;
                next_start = fista  ([],@(x) combined_grad(x),[] ,Hfun_prox ,lambda_H,next_start,par) ;
            end
            %restore values for last iteration
            startx = next_start ;
            mu = save_mu ;
            par.eps = save_eps ;
        end
        
        out =fista  ( [], @(x) combined_grad(x),[] ,Hfun_prox ,lambda_H,startx,par) ;
    end
else % all function handles are given
    if (par.continuation)
        save_mu = mu ;
        save_eps = par.eps ;
        save_print_flag = par.print_flag ;
        next_start = startx ;
        par.eps = par.eps * mult_eps_const ; %inner cycles with bigger eps
        par.print_flag = false ; %inner cycles with silence printing
        if (save_print_flag)
            fprintf('*********************\n');
            fprintf('Starting continuation cycles\n')
        end
        for k = par.cont_cycles-1:-1:1
            mu =  save_mu * par.gamma ^ k ;
            [next_start,~,parout] = fista  (@(x) combinedf(x),@(x) combined_grad(x),Hfun ,Hfun_prox ,lambda_H,next_start,par) ;
            if (save_print_flag)
                fprintf('Cycle %d (mu = %g) required %d iterations\n',par.cont_cycles - k,mu,parout.iterNum) ;
            end
            CycleIter(par.cont_cycles - k) =  parout.iterNum ;
        end
        %restore values for last iteration
        startx = next_start ;
        mu = save_mu ;
        par.eps = save_eps ;
        par.print_flag = save_print_flag ;
    end
    
    if (par.print_flag && par.continuation)
        fprintf('Starting final cycle with mu = %g\n',mu) ;
    end
    [out,fmin,parout] =fista  (@(x) combinedf(x),@(x) combined_grad(x),Hfun ,Hfun_prox ,lambda_H,startx,par) ;
    if ((nargout == 3 ) || par.print_flag)
        %calculation original function value
        OrigFun = Ffun(out) + lambda_G * Gfun(Afun(out)) + lambda_H * Hfun(out) ;
    end
    if (nargout == 3)
        if (par.continuation)
            CycleIter(par.cont_cycles) =  parout.iterNum ;
            parout.CyclIter = CycleIter ;
        end
        parout.OrigFun = OrigFun ;
    end
    if (par.print_flag)
        fprintf('Original function value = %15f \n',OrigFun) ;
    end

end


end

